<?php
/**
 * ASSET CODE HELPER
 * 
 * Helper class untuk auto-generate kode asset saat insert data baru
 * 
 * CARA PENGGUNAAN:
 * 
 * // Saat insert data baru di form_tambah_inventory.php atau sejenisnya:
 * 
 * require_once __DIR__ . '/AssetCodeHelper.php';
 * 
 * // Setelah INSERT data inventory berhasil:
 * $insert_id = $conn->insert_id;
 * 
 * // Auto-generate kode asset
 * $codeHelper = new AssetCodeHelper($conn);
 * $kode_asset = $codeHelper->autoGenerate($insert_id);
 * 
 * if ($kode_asset) {
 *     echo "Kode Asset: " . $kode_asset;
 * }
 * 
 */

class AssetCodeHelper {
    private $conn;
    
    // MAPPING KODE KANTOR
    private $kode_kantor = [
        'Pusat' => 'PST',
        'Cabang Jakarta' => 'CJK',
        'Cabang Bandung' => 'CBD',
        'Cabang Surabaya' => 'CSB',
        'Cabang Medan' => 'CMD',
        'Cabang Yogyakarta' => 'CYK',
        'Cabang Semarang' => 'CSM',
        'Cabang Bali' => 'CBA',
        'Kantor Pusat' => 'PST',
        'Jakarta' => 'JKT',
        'Bandung' => 'BDG',
        'Surabaya' => 'SBY',
        'Medan' => 'MDN',
        'Yogyakarta' => 'YGK',
        'Semarang' => 'SMG',
        'Bali' => 'BAL',
    ];
    
    // MAPPING KODE KATEGORI
    private $kode_kategori = [
        'Elektronik' => 'ELK',
        'Furniture' => 'FRN',
        'Kendaraan' => 'KDN',
        'Komputer' => 'KMP',
        'Alat Tulis' => 'ATK',
        'Peralatan Kantor' => 'PKT',
        'Bangunan' => 'BGN',
        'Peralatan Teknologi' => 'TEK',
        'Alat Kesehatan' => 'KSH',
        'Peralatan Olahraga' => 'OLG',
        'Perlengkapan' => 'PLG',
        'Mesin' => 'MSN',
        'Perangkat' => 'PRG',
    ];
    
    // MAPPING KODE SUMBER DANA
    private $kode_sumber_dana = [
        'BOS' => 'BOS',
        'Hibah' => 'HBH',
        'Yayasan' => 'YYS',
        'APBN' => 'ABN',
        'APBD' => 'ABD',
        'Swadaya' => 'SWD',
        'Dana Desa' => 'DDS',
        'CSR' => 'CSR',
        'Donasi' => 'DNS',
        'Pembelian' => 'BLI',
        'Bantuan' => 'BTN',
    ];
    
    public function __construct($connection) {
        $this->conn = $connection;
    }
    
    private function getKodeKantor($nama_kantor) {
        if (isset($this->kode_kantor[$nama_kantor])) {
            return $this->kode_kantor[$nama_kantor];
        }
        
        $words = explode(' ', $nama_kantor);
        if (count($words) >= 2) {
            $kode = '';
            foreach (array_slice($words, 0, 3) as $word) {
                if (!empty($word)) {
                    $kode .= strtoupper(substr($word, 0, 1));
                }
            }
            return str_pad($kode, 3, 'X');
        }
        
        return strtoupper(substr($nama_kantor, 0, 3));
    }
    
    private function getKodeKategori($nama_kategori) {
        if (isset($this->kode_kategori[$nama_kategori])) {
            return $this->kode_kategori[$nama_kategori];
        }
        
        $words = explode(' ', $nama_kategori);
        if (count($words) >= 2) {
            $kode = '';
            foreach (array_slice($words, 0, 3) as $word) {
                if (!empty($word)) {
                    $kode .= strtoupper(substr($word, 0, 1));
                }
            }
            return str_pad($kode, 3, 'X');
        }
        
        return strtoupper(substr($nama_kategori, 0, 3));
    }
    
    private function getKodeSumberDana($nama_sumber) {
        if (isset($this->kode_sumber_dana[$nama_sumber])) {
            return $this->kode_sumber_dana[$nama_sumber];
        }
        
        return strtoupper(substr($nama_sumber, 0, 3));
    }
    
    private function getKodeBarang($kantor, $kategori, $tahun = null) {
        if (!$tahun) {
            $tahun = date('Y');
        }
        
        $sql = "SELECT COUNT(*) as total FROM inventory 
                WHERE kantor = ? 
                AND kategori = ? 
                AND YEAR(tanggal_perolehan) = ?";
        
        $stmt = $this->conn->prepare($sql);
        
        if (!$stmt) {
            return '001';
        }
        
        $stmt->bind_param("sss", $kantor, $kategori, $tahun);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result && $row = $result->fetch_assoc()) {
            $counter = $row['total'];
        } else {
            $counter = 1;
        }
        
        $stmt->close();
        return str_pad($counter, 3, '0', STR_PAD_LEFT);
    }
    
    /**
     * Auto-generate kode asset setelah insert data baru
     * 
     * @param int $insert_id - ID yang baru di-insert
     * @return string|null - Kode asset yang digenerate atau null jika gagal
     */
    public function autoGenerate($insert_id) {
        // Ambil data yang baru diinsert
        $sql = "SELECT * FROM inventory WHERE id = ?";
        $stmt = $this->conn->prepare($sql);
        
        if (!$stmt) return null;
        
        $stmt->bind_param("i", $insert_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $data = $result->fetch_assoc();
        $stmt->close();
        
        if (!$data) return null;
        
        // Validasi data wajib
        if (empty($data['kantor']) || empty($data['kategori']) || empty($data['sumber_dana'])) {
            return null;
        }
        
        // Generate komponen kode
        $kode_k = $this->getKodeKantor($data['kantor']);
        $kode_kat = $this->getKodeKategori($data['kategori']);
        
        $tahun = isset($data['tanggal_perolehan']) ? date('Y', strtotime($data['tanggal_perolehan'])) : date('Y');
        $kode_brg = $this->getKodeBarang($data['kantor'], $data['kategori'], $tahun);
        
        $kode_sd = $this->getKodeSumberDana($data['sumber_dana']);
        
        // Format final
        $kode_asset = "{$kode_k}.{$kode_kat}.{$kode_brg}.{$kode_sd}";
        
        // Update ke database
        $sql_update = "UPDATE inventory SET kode_aset_auto = ? WHERE id = ?";
        $stmt_update = $this->conn->prepare($sql_update);
        
        if (!$stmt_update) return null;
        
        $stmt_update->bind_param("si", $kode_asset, $insert_id);
        $success = $stmt_update->execute();
        $stmt_update->close();
        
        return $success ? $kode_asset : null;
    }
}
?>