<?php
session_start();
require_once __DIR__ . '/../../DataBase/koneksi.php';
require_once __DIR__ . '/../../config/RencanaKerjaPermission.php';
require_once __DIR__ . '/../../config/session_check.php';

// Define base path
if (!defined('BASE_PATH')) {
    define('BASE_PATH', '/inventoryACF');
}

// Validate role using helper
requireRole(['admin', 'director', 'dept_head', 'staff']);

$userRole = strtolower(trim($_SESSION['role']));
$userId = (int) $_SESSION['user_id'];
$userDept = $_SESSION['department'] ?? null;
$permission = new RencanaKerjaPermission($conn, $_SESSION['user_id'], $userRole);

// IMPORTANT: Manager must have department
if ($userRole === 'dept_head' && empty($userDept)) {
    die("
    <!DOCTYPE html>
    <html>
    <head>
        <meta charset='UTF-8'>
        <title>Error - Department Required</title>
        <script src='https://cdn.tailwindcss.com'></script>
    </head>
    <body class='bg-gray-50'>
        <div class='min-h-screen flex items-center justify-center p-4'>
            <div class='bg-white p-8 rounded-xl shadow-lg max-w-md w-full border-l-4 border-red-500'>
                <div class='flex items-center gap-3 mb-4'>
                    <div class='w-12 h-12 bg-red-100 rounded-full flex items-center justify-center'>
                        <svg class='w-6 h-6 text-red-600' fill='none' stroke='currentColor' viewBox='0 0 24 24'>
                            <path stroke-linecap='round' stroke-linejoin='round' stroke-width='2' d='M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z'></path>
                        </svg>
                    </div>
                    <h1 class='text-xl font-bold text-gray-900'>Department Required</h1>
                </div>
                <p class='text-gray-600 mb-4'>Manager harus memiliki department yang ditugaskan untuk dapat membuat rencana kerja.</p>
                <p class='text-sm text-gray-500 mb-6'>Silakan hubungi administrator sistem untuk mengatur department Anda.</p>
                <button onclick='goBackToDashboard()' class='w-full bg-gradient-to-r from-blue-500 to-blue-600 hover:from-blue-600 hover:to-blue-700 text-white font-semibold py-3 px-6 rounded-lg transition-all duration-200 transform hover:scale-105'>
                    Kembali ke Dashboard
                </button>
            </div>
        </div>
        <script>
            function goBackToDashboard() {
                if (parent && parent.document.getElementById('contentFrame')) {
                    parent.document.getElementById('contentFrame').src = '" . BASE_PATH . "/dashboard.php';
                } else {
                    window.location.href = '" . BASE_PATH . "/dashboard.php';
                }
            }
        </script>
    </body>
    </html>
    ");
}

// Handle form submission
$errors = [];
$successRedirect = false;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Validate CSRF Token
    if (!isset($_POST['csrf_token']) || !validateCsrfToken($_POST['csrf_token'])) {
        $errors[] = "Token keamanan tidak valid. Silakan refresh halaman dan coba lagi.";
    } else {
        $judulTugas = trim($_POST['judul_tugas'] ?? '');
        $deskripsi = trim($_POST['deskripsi'] ?? '');
        $department = trim($_POST['department'] ?? '');
        $assignedTo = (int)($_POST['assigned_to'] ?? 0);
        $tenggatWaktu = trim($_POST['tenggat_waktu'] ?? '');
        $priority = trim($_POST['priority'] ?? 'penting_tidak_genting');
        $status = 'pending';
        $progress = 0;

        // ✅ STAFF: Force assign to self
        if ($userRole === 'staff') {
            $assignedTo = $userId;
        }

        // Validation
        if (empty($judulTugas)) $errors[] = "Judul tugas harus diisi";
        if (empty($department)) $errors[] = "Department harus dipilih";
        if (empty($assignedTo)) $errors[] = "Harus menentukan penerima tugas";
        if (empty($tenggatWaktu)) $errors[] = "Tenggat waktu harus diisi";
        
        // MANAGER VALIDATION: Can only assign to their own department
        if ($userRole === 'dept_head' && $department !== $userDept) {
            $errors[] = "Manager hanya bisa membuat tugas untuk department sendiri ({$userDept})";
        }

        // Validate deadline is not in the past
        $deadline = strtotime($tenggatWaktu);
        $today = strtotime(date('Y-m-d'));
        if ($deadline < $today) {
            $errors[] = "Tenggat waktu tidak boleh di masa lalu";
        }

        if (empty($errors)) {
            $sql = "INSERT INTO rencana_kerja 
                    (judul_tugas, deskripsi, department, assigned_to, created_by, tenggat_waktu, priority, status, progress, created_at) 
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())";
            
            $stmt = $conn->prepare($sql);
            
            if (!$stmt) {
                $errors[] = "Database error: " . $conn->error;
            } else {
                $stmt->bind_param("sssiisssi", $judulTugas, $deskripsi, $department, $assignedTo, $userId, $tenggatWaktu, $priority, $status, $progress);
                
                if ($stmt->execute()) {
                    $_SESSION['success_message'] = "Rencana kerja berhasil ditambahkan";
                    $successRedirect = true;
                } else {
                    $errors[] = "Gagal menyimpan data: " . $stmt->error;
                }
                
                $stmt->close();
            }
        }
    }
}

// Get departments first
$departments = [];

// ✅ STAFF: Auto-populate with their department
if ($userRole === 'staff') {
    if (!empty($userDept)) {
        $departments = [$userDept];
    }
} else {
    $sqlDept = "SELECT DISTINCT department FROM users WHERE department IS NOT NULL AND department != '' AND status = 'active' ORDER BY department ASC";
    $resultDept = $conn->query($sqlDept);
    if ($resultDept && $resultDept->num_rows > 0) {
        while ($row = $resultDept->fetch_assoc()) {
            $departments[] = $row['department'];
        }
    }
}

// Get users (only for non-staff roles)
$users = [];
if ($userRole === 'staff') {
    // Staff: No need to get users list
    $users = [];
} elseif ($userRole === 'dept_head') {
    // Manager: only their department users
    $departments = [$userDept];
    
    $sqlUsers = "SELECT id, username, full_name, email, department, role 
                 FROM users 
                 WHERE department = ? 
                 AND status = 'active' 
                 AND is_active = 1
                 ORDER BY full_name ASC";
    
    $stmtUsers = $conn->prepare($sqlUsers);
    
    if ($stmtUsers) {
        $stmtUsers->bind_param("s", $userDept);
        $stmtUsers->execute();
        $resultUsers = $stmtUsers->get_result();
        
        while ($row = $resultUsers->fetch_assoc()) {
            $users[] = [
                'id' => $row['id'],
                'full_name' => $row['full_name'] ?? $row['username'],
                'email' => $row['email'],
                'department' => $row['department'],
                'role' => $row['role']
            ];
        }
        $stmtUsers->close();
    }
} else {
    // Admin/Director: all users
    $sqlUsers = "SELECT id, username, full_name, email, department, role 
                 FROM users 
                 WHERE status = 'active' 
                 AND is_active = 1
                 ORDER BY department ASC, full_name ASC";
    
    $resultUsers = $conn->query($sqlUsers);
    if ($resultUsers && $resultUsers->num_rows > 0) {
        while ($row = $resultUsers->fetch_assoc()) {
            $users[] = [
                'id' => $row['id'],
                'full_name' => $row['full_name'] ?? $row['username'],
                'email' => $row['email'],
                'department' => $row['department'],
                'role' => $row['role']
            ];
        }
    }
}

// Dynamic back URL based on role
$backUrl = match($userRole) {
    'admin' => 'pages/admin/rencana_kerja.php',
    'dept_head' => 'pages/manager/rencana_kerja.php',
    'director' => 'pages/director/rencana_kerja.php',
    'staff' => 'pages/staff/rencana_kerja.php',
    default => 'pages/staff/rencana_kerja.php'
};

// Generate CSRF Token
$csrfToken = generateCsrfToken();
?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Tambah Rencana Kerja</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    colors: {
                        primary: '#14b8a6',
                        'primary-dark': '#0f766e',
                        'primary-light': '#5eead4',
                    },
                    fontFamily: {
                        sans: ['Inter', 'sans-serif'],
                    },
                }
            }
        }
    </script>
    <style>
        body {
            font-family: 'Inter', sans-serif;
            background: #f7f7f5;
        }
    </style>
    
    <?php if ($successRedirect): ?>
    <script>
        (function() {
            const targetUrl = '<?= $backUrl ?>';
            
            if (parent && parent.document.getElementById('contentFrame')) {
                parent.document.getElementById('contentFrame').src = targetUrl;
            } else {
                window.location.href = '<?= BASE_PATH ?>/dashboard.php?load=' + encodeURIComponent(targetUrl);
            }
        })();
    </script>
    <?php endif; ?>
</head>
<body>
    <div class="min-h-screen py-4">
        <div class="max-w-3xl mx-auto px-4">
            <!-- Header -->
            <div class="mb-3">
                <div class="flex items-center gap-2 mb-2">
                    <button 
                        type="button"
                        onclick="goBack('<?= $backUrl ?>')"
                        class="w-8 h-8 flex items-center justify-center rounded-lg bg-white hover:bg-slate-50 border border-slate-200 transition text-sm cursor-pointer">
                        &larr;
                    </button>

                    <div>
                        <div class="flex items-center gap-2">
                            <h1 class="text-lg font-bold text-slate-900">Tambah Rencana Kerja Baru</h1>
                            <?php if ($userRole === 'dept_head'): ?>
                                <span class="px-2 py-0.5 bg-green-100 text-green-700 rounded-full text-xs font-semibold border border-green-200">
                                    <?= htmlspecialchars($userDept) ?>
                                </span>
                            <?php elseif ($userRole === 'staff'): ?>
                                <span class="px-2 py-0.5 bg-blue-100 text-blue-700 rounded-full text-xs font-semibold border border-blue-200">
                                    Personal Task
                                </span>
                            <?php endif; ?>
                        </div>
                        <p class="text-xs text-slate-600">
                            <?= $userRole === 'staff' ? 'Buat tugas untuk diri sendiri' : 'Buat tugas baru untuk tim Anda' ?>
                        </p>
                    </div>
                </div>
            </div>

            <!-- Error Messages -->
            <?php if (!empty($errors)): ?>
            <div class="bg-red-50 border-l-4 border-red-500 rounded-lg p-3 mb-3 shadow-sm">
                <div class="flex items-start gap-2">
                    <div class="w-7 h-7 bg-red-500 rounded-lg flex items-center justify-center flex-shrink-0 text-white text-xs font-bold">
                        !
                    </div>
                    <div class="flex-1">
                        <h3 class="font-semibold text-red-900 text-sm mb-1">Terdapat kesalahan:</h3>
                        <ul class="list-disc list-inside text-red-700 space-y-0.5 text-xs">
                            <?php foreach ($errors as $error): ?>
                                <li><?= htmlspecialchars($error) ?></li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                </div>
            </div>
            <?php endif; ?>

            <!-- Form -->
            <form method="POST" class="bg-white rounded-lg shadow-sm border border-slate-200" onsubmit="return validateForm()">
                <input type="hidden" name="csrf_token" value="<?= htmlspecialchars($csrfToken) ?>">
                
                <div class="p-4 space-y-3">
                    <!-- Judul Tugas -->
                    <div>
                        <label for="judul_tugas" class="block text-xs font-semibold text-slate-700 mb-1">
                            Judul Tugas <span class="text-red-500">*</span>
                        </label>
                        <input type="text" id="judul_tugas" name="judul_tugas" required
                               value="<?= htmlspecialchars($_POST['judul_tugas'] ?? '') ?>"
                               maxlength="255"
                               class="w-full px-3 py-2 text-sm border border-slate-200 rounded-lg focus:ring-2 focus:ring-primary focus:border-primary transition"
                               placeholder="Contoh: Membuat laporan bulanan...">
                    </div>

                    <!-- Deskripsi -->
                    <div>
                        <label for="deskripsi" class="block text-xs font-semibold text-slate-700 mb-1">
                            Deskripsi
                        </label>
                        <textarea id="deskripsi" name="deskripsi" rows="3"
                                  maxlength="1000"
                                  class="w-full px-3 py-2 text-sm border border-slate-200 rounded-lg focus:ring-2 focus:ring-primary focus:border-primary transition"
                                  placeholder="Jelaskan detail tugas yang harus dikerjakan..."><?= htmlspecialchars($_POST['deskripsi'] ?? '') ?></textarea>
                        <p class="text-xs text-slate-500 mt-1" id="char-count">0 / 1000 karakter</p>
                    </div>

                    <!-- Department & Assigned To -->
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-3">
                        <div>
                            <label for="department" class="block text-xs font-semibold text-slate-700 mb-1">
                                Department <span class="text-red-500">*</span>
                            </label>
                            <?php if ($userRole === 'dept_head' || $userRole === 'staff'): ?>
                                <!-- Dept Head & Staff: Fixed department -->
                                <input type="text" 
                                       value="<?= htmlspecialchars($userDept ?? 'Tidak Ada Department') ?>" 
                                       disabled
                                       class="w-full px-3 py-2 text-sm border border-slate-200 rounded-lg bg-slate-50 text-slate-700 cursor-not-allowed">
                                <input type="hidden" name="department" value="<?= htmlspecialchars($userDept ?? '') ?>">
                                <p class="text-xs text-slate-500 mt-1">
                                    <?= $userRole === 'dept_head' ? 'Manager hanya dapat membuat tugas untuk department sendiri' : 'Department Anda' ?>
                                </p>
                            <?php else: ?>
                                <!-- Admin/Director: Can select department -->
                                <select id="department" name="department" required
                                        class="w-full px-3 py-2 text-sm border border-slate-200 rounded-lg focus:ring-2 focus:ring-primary focus:border-primary transition bg-white"
                                        onchange="filterUsersByDepartment(this.value)">
                                    <option value="">Pilih Department</option>
                                    <?php foreach ($departments as $dept): ?>
                                        <option value="<?= htmlspecialchars($dept) ?>" 
                                                <?= (($_POST['department'] ?? '') === $dept) ? 'selected' : '' ?>>
                                            <?= htmlspecialchars($dept) ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            <?php endif; ?>
                        </div>

                        <div>
                            <label for="assigned_to" class="block text-xs font-semibold text-slate-700 mb-1">
                                Ditugaskan Kepada <span class="text-red-500">*</span>
                            </label>
                            <?php if ($userRole === 'staff'): ?>
                                <!-- ✅ Staff: Auto-assigned to self -->
                                <input type="text" 
                                       value="<?= htmlspecialchars($_SESSION['full_name'] ?? $_SESSION['username'] ?? 'Diri Sendiri') ?>" 
                                       disabled
                                       class="w-full px-3 py-2 text-sm border border-slate-200 rounded-lg bg-slate-50 text-slate-700 cursor-not-allowed">
                                <input type="hidden" name="assigned_to" value="<?= $userId ?>">
                                <p class="text-xs text-slate-500 mt-1">
                                    Staff hanya dapat membuat tugas untuk diri sendiri
                                </p>
                            <?php else: ?>
                                <!-- Other roles: Can select users -->
                                <select id="assigned_to" name="assigned_to" required
                                        class="w-full px-3 py-2 text-sm border border-slate-200 rounded-lg focus:ring-2 focus:ring-primary focus:border-primary transition bg-white">
                                    <option value="">Pilih User</option>
                                    <?php if (empty($users)): ?>
                                        <option value="" disabled>Tidak ada user tersedia</option>
                                    <?php else: ?>
                                        <?php foreach ($users as $user): ?>
                                            <option value="<?= $user['id'] ?>"
                                                    data-department="<?= htmlspecialchars($user['department'] ?? '') ?>"
                                                    <?= (($_POST['assigned_to'] ?? '') == $user['id']) ? 'selected' : '' ?>>
                                                <?= htmlspecialchars($user['full_name'] ?? 'Unknown') ?> 
                                                <?php if ($userRole !== 'dept_head'): ?>
                                                    (<?= htmlspecialchars($user['department'] ?? 'N/A') ?>)
                                                <?php endif; ?>
                                            </option>
                                        <?php endforeach; ?>
                                    <?php endif; ?>
                                </select>
                                <?php if ($userRole === 'dept_head'): ?>
                                    <?php if (count($users) > 0): ?>
                                        <p class="text-xs text-slate-500 mt-1">
                                            Menampilkan <?= count($users) ?> anggota tim di <?= htmlspecialchars($userDept) ?>
                                        </p>
                                    <?php else: ?>
                                        <p class="text-xs text-red-600 mt-1">
                                            Tidak ada user aktif di department "<?= htmlspecialchars($userDept) ?>". Hubungi administrator.
                                        </p>
                                    <?php endif; ?>
                                <?php endif; ?>
                            <?php endif; ?>
                        </div>
                    </div>

                    <!-- Tenggat Waktu & Priority -->
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-3">
                        <div>
                            <label for="tenggat_waktu" class="block text-xs font-semibold text-slate-700 mb-1">
                                Tenggat Waktu <span class="text-red-500">*</span>
                            </label>
                            <input type="date" id="tenggat_waktu" name="tenggat_waktu" required
                                   value="<?= htmlspecialchars($_POST['tenggat_waktu'] ?? '') ?>"
                                   min="<?= date('Y-m-d') ?>"
                                   class="w-full px-3 py-2 text-sm border border-slate-200 rounded-lg focus:ring-2 focus:ring-primary focus:border-primary transition">
                        </div>

                        <div>
                            <label for="priority" class="block text-xs font-semibold text-slate-700 mb-1">
                                Prioritas
                            </label>
                            <select id="priority" name="priority"
                                    class="w-full px-3 py-2 text-sm border border-slate-200 rounded-lg focus:ring-2 focus:ring-primary focus:border-primary transition bg-white">
                                <option value="penting_genting" <?= (($_POST['priority'] ?? 'penting_tidak_genting') === 'penting_genting') ? 'selected' : '' ?>>
                                    Penting & Genting
                                </option>
                                <option value="penting_tidak_genting" <?= (($_POST['priority'] ?? 'penting_tidak_genting') === 'penting_tidak_genting') ? 'selected' : '' ?>>
                                    Penting Tidak Genting
                                </option>
                                <option value="penting" <?= (($_POST['priority'] ?? 'penting_tidak_genting') === 'penting') ? 'selected' : '' ?>>
                                    Penting
                                </option>
                                <option value="tidak_penting" <?= (($_POST['priority'] ?? 'penting_tidak_genting') === 'tidak_penting') ? 'selected' : '' ?>>
                                    Tidak Penting
                                </option>
                            </select>
                        </div>
                    </div>

                    <!-- Info Note -->
                    <div class="bg-blue-50 border-l-4 border-blue-500 rounded-lg p-2.5">
                        <div class="flex items-start gap-2">
                            <div class="w-5 h-5 bg-blue-500 rounded-lg flex items-center justify-center flex-shrink-0 text-white text-xs font-bold">
                                i
                            </div>
                            <div class="text-xs text-blue-800">
                                <p class="font-semibold mb-0.5">Informasi:</p>
                                <ul class="list-disc list-inside space-y-0.5">
                                    <li>Tugas baru akan otomatis berstatus <strong>"Pending"</strong></li>
                                    <li>Progress awal dimulai dari <strong>0%</strong></li>
                                    <li>Pastikan memilih tenggat waktu yang realistis</li>
                                    <?php if ($userRole === 'staff'): ?>
                                        <li>Tugas ini akan ditugaskan kepada <strong>Anda sendiri</strong></li>
                                    <?php elseif ($userRole === 'dept_head'): ?>
                                        <li>Anda hanya dapat membuat tugas untuk department <strong><?= htmlspecialchars($userDept) ?></strong></li>
                                    <?php else: ?>
                                        <li>Filter user akan otomatis berdasarkan department yang dipilih</li>
                                    <?php endif; ?>
                                </ul>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Form Footer -->
                <div class="bg-slate-50 px-4 py-2.5 rounded-b-lg flex items-center justify-end gap-2 border-t border-slate-200">
                    <button 
                        type="button" 
                        onclick="goBack('<?= $backUrl ?>')" 
                        class="px-4 py-2 text-sm border border-slate-300 text-slate-700 rounded-lg hover:bg-white transition font-medium cursor-pointer">
                        Batal
                    </button>
                    <button type="submit" class="px-4 py-2 text-sm bg-gradient-to-r from-primary to-primary-dark hover:shadow-lg text-white rounded-lg transition font-medium">
                        Tambah Tugas
                    </button>
                </div>
            </form>
        </div>
    </div>

    <script>
        const BASE_PATH = '<?= BASE_PATH ?>';
        const userRole = '<?= $userRole ?>';
        
        function goBack(url) {
            if (parent && parent.document.getElementById('contentFrame')) {
                parent.document.getElementById('contentFrame').src = url;
                return;
            }
            
            window.location.href = BASE_PATH + '/dashboard.php?load=' + encodeURIComponent(url);
        }
        
        // Character counter for description
        const deskripsi = document.getElementById('deskripsi');
        const charCount = document.getElementById('char-count');
        
        if (deskripsi && charCount) {
            deskripsi.addEventListener('input', function() {
                const count = this.value.length;
                charCount.textContent = `${count} / 1000 karakter`;
                
                if (count > 900) {
                    charCount.classList.add('text-orange-600');
                } else {
                    charCount.classList.remove('text-orange-600');
                }
            });
            
            deskripsi.dispatchEvent(new Event('input'));
        }
        
        // Filter users by department (only for non-Manager and non-Staff roles)
        function filterUsersByDepartment(selectedDept) {
            if (userRole === 'dept_head' || userRole === 'staff') {
                return;
            }
            
            const userSelect = document.getElementById('assigned_to');
            if (!userSelect) return;
            
            const options = userSelect.querySelectorAll('option');
            
            options.forEach(option => {
                if (option.value === '') {
                    option.style.display = 'block';
                    return;
                }
                
                const userDept = option.getAttribute('data-department');
                if (selectedDept === '' || userDept === selectedDept) {
                    option.style.display = 'block';
                } else {
                    option.style.display = 'none';
                }
            });
            
            const currentOption = userSelect.options[userSelect.selectedIndex];
            if (currentOption && currentOption.style.display === 'none') {
                userSelect.value = '';
            }
        }

        // Form validation before submit
        function validateForm() {
            const judulTugas = document.getElementById('judul_tugas').value.trim();
            const tenggatWaktu = document.getElementById('tenggat_waktu').value;
            
            if (!judulTugas) {
                alert('Judul tugas harus diisi!');
                return false;
            }
            
            if (!tenggatWaktu) {
                alert('Tenggat waktu harus diisi!');
                return false;
            }
            
            // Staff role: assigned_to is auto-filled
            if (userRole !== 'staff') {
                const assignedTo = document.getElementById('assigned_to').value;
                if (!assignedTo) {
                    alert('Harus menentukan penerima tugas!');
                    return false;
                }
            }
            
            const deadline = new Date(tenggatWaktu);
            const today = new Date();
            today.setHours(0, 0, 0, 0);
            
            if (deadline < today) {
                alert('Tenggat waktu tidak boleh di masa lalu!');
                return false;
            }
            
            return true;
        }

        // Page initialization
        document.addEventListener('DOMContentLoaded', function() {
            const today = new Date().toISOString().split('T')[0];
            document.getElementById('tenggat_waktu').setAttribute('min', today);
            
            console.log('User Role:', userRole);
            <?php if ($userRole === 'dept_head'): ?>
            console.log('Manager Department:', '<?= htmlspecialchars($userDept) ?>');
            console.log('Available Users:', <?= count($users) ?>);
            <?php elseif ($userRole === 'staff'): ?>
            console.log('Staff ID:', <?= $userId ?>);
            console.log('Auto-assigned to self');
            <?php endif; ?>
        });
    </script>
</body>
</html>